<?php
/* --------------------------------------------------------------
   ApiV2Authenticator.inc.php 2019-11-11
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

class ApiV2Authenticator
{
    /**
     * @var Request
     */
    protected $request;
    
    /**
     * @var Response
     */
    protected $response;
    
    /**
     * @var string
     */
    protected $method;
    
    /**
     * @var array
     */
    protected $uri;
    
    
    /**
     * ApiV2Authenticator constructor.
     *
     * @param Request  $request
     * @param Response $response
     * @param array    $uri
     */
    public function __construct(Request $request, Response $response, array $uri)
    {
        $this->request  = $request;
        $this->response = $response;
        $this->method   = strtoupper($request->getMethod());
        $this->uri      = $uri;
    }
    
    
    /**
     * Authorize request with HTTP Basic Authorization
     *
     * Call this method in every API operation that needs to be authorized with the HTTP Basic
     * Authorization technique.
     *
     * @link http://php.net/manual/en/features.http-auth.php
     *
     * Not available to child-controllers (private method).
     *
     * @param string $controllerName Name of the parent controller for this api call.
     *
     * @throws HttpApiV2Exception If request does not provide the "Authorization" header or if the
     *                            credentials are invalid.
     *
     * @throws InvalidArgumentException If the username or password values are invalid.
     * @throws JsonWebTokenException If a JWT supplied via “Authorization: Bearer” is found to be invalid
     */
    public function authorize($controllerName)
    {
        if (!empty($_SERVER['HTTP_AUTHORIZATION']) && strpos($_SERVER['HTTP_AUTHORIZATION'], 'Bearer') !== false) {
            $this->authorizeBearer($controllerName);
        } else {
            $this->authorizeBasicAuth($controllerName);
        }
    }
    
    
    /**
     * Authorizes requests by Basic Auth.
     *
     * @param $controllerName
     *
     * @throws HttpApiV2Exception
     */
    protected function authorizeBasicAuth($controllerName)
    {
        if (empty($_SERVER['PHP_AUTH_USER']) && empty($_SERVER['PHP_AUTH_PW'])
            && !empty($_SERVER['HTTP_AUTHORIZATION'])) {
            list($_SERVER['PHP_AUTH_USER'], $_SERVER['PHP_AUTH_PW']) = explode(':',
                                                                               base64_decode(substr($_SERVER['HTTP_AUTHORIZATION'],
                                                                                                    6)));
        } elseif (empty($_SERVER['PHP_AUTH_USER']) && empty($_SERVER['PHP_AUTH_PW'])
                  && !empty($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            list($_SERVER['PHP_AUTH_USER'], $_SERVER['PHP_AUTH_PW']) = explode(':',
                                                                               base64_decode(substr($_SERVER['REDIRECT_HTTP_AUTHORIZATION'],
                                                                                                    6)));
        }
        
        if (!isset($_SERVER['PHP_AUTH_USER'])) {
            $this->response = $this->response->withHeader('WWW-Authenticate', 'Basic realm="Gambio GX3 APIv2 Login"');
            throw new HttpApiV2Exception('Unauthorized', 401);
        }
        
        $authService = StaticGXCoreLoader::getService('Auth');
        $credentials = MainFactory::create('UsernamePasswordCredentials',
                                           new NonEmptyStringType($_SERVER['PHP_AUTH_USER']),
                                           new StringType($_SERVER['PHP_AUTH_PW']));
        
        $db      = StaticGXCoreLoader::getDatabaseQueryBuilder();
        $query   = $db->get_where('customers',
                                  [
                                      'customers_email_address' => $_SERVER['PHP_AUTH_USER'],
                                      'customers_status'        => '0'
                                  ]);
        $isAdmin = $query->num_rows() === 1;
        $user    = $query->row_array();
        
        if (!$isAdmin || !$authService->authUser($credentials)) {
            throw new HttpApiV2Exception('Invalid Credentials', 401);
        }
        
        $controllerName     = substr($controllerName, 0, -10);
        $adminAccessService = StaticGXCoreLoader::getService('AdminAccess');
        $hasPermission      = (bool)$adminAccessService->checkReadingPermissionForController(new NonEmptyStringType('DefaultApiV2'),
                                                                                             new IdType((int)$user['customers_id']));
        
        if (($this->request->isPost() && $this->uri[count($this->uri) - 1] !== 'search')
            || $this->request->isPut()
            || $this->request->isPatch()) {
            $hasPermission &= (bool)$adminAccessService->checkWritingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                             new IdType((int)$user['customers_id']));
        } elseif ($this->request->isDelete()) {
            $hasPermission &= (bool)$adminAccessService->checkDeletingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                              new IdType((int)$user['customers_id']));
        } else {
            $hasPermission &= (bool)$adminAccessService->checkReadingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                             new IdType((int)$user['customers_id']));
        }
        
        if (!$hasPermission) {
            throw new HttpApiV2Exception('Forbidden - No Permissions', 403);
        }
        // authorization valid
    }
    
    
    /**
     * Authorize requests with JWT header.
     *
     * @param $controllerName
     *
     * @throws HttpApiV2Exception
     * @throws JsonWebTokenException
     */
    protected function authorizeBearer($controllerName)
    {
        list($bearer, $token) = explode(' ', $_SERVER['HTTP_AUTHORIZATION']);
        if ($bearer !== 'Bearer' || empty($token)) {
            throw new HttpApiV2Exception('Invalid syntax in Authorization header');
        }
        
        $secret      = MainFactory::create('NonEmptyStringType', JsonWebTokenSecretProvider::getSecret());
        $tokenString = MainFactory::create('NonEmptyStringType', $token);
        $parsedToken = JsonWebTokenParser::parseToken($tokenString, $secret);
        
        $customerId         = new IdType((int)$parsedToken->getPayload()->getValue('customer_id'));
        $adminAccessService = StaticGXCoreLoader::getService('AdminAccess');
        $issTimestamp       = abs((int)$parsedToken->getPayload()->getValue('iat'));
        $expTimestamp       = abs((int)$parsedToken->getPayload()->getValue('exp'));
        $currentTime        = time();
        $hasPermission      = (bool)$adminAccessService->checkReadingPermissionForController(new NonEmptyStringType('DefaultApiV2'),
                                                                                             $customerId);
        if (($this->request->isPost() && $this->uri[count($this->uri) - 1] !== 'search')
            || $this->request->isPut()
            || $this->request->isPatch()) {
            $hasPermission &= (bool)$adminAccessService->checkWritingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                             $customerId);
        } elseif ($this->request->isDelete()) {
            $hasPermission &= (bool)$adminAccessService->checkDeletingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                              $customerId);
        } else {
            $hasPermission &= (bool)$adminAccessService->checkReadingPermissionForController(new NonEmptyStringType($controllerName),
                                                                                             $customerId);
        }
        if ($issTimestamp === 0 || $expTimestamp === 0) {
            throw new JsonWebTokenException('invalid exp/iat in token');
        }
        $timeValid = $issTimestamp <= $currentTime && $currentTime <= $expTimestamp;
        
        if (!$hasPermission || !$timeValid) {
            throw new JsonWebTokenException('permission denied');
        }
        // authorization valid
    }
}
